package pmf.imi;

import io.github.bonigarcia.wdm.WebDriverManager;
import org.checkerframework.checker.units.qual.C;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.edge.EdgeDriver;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.testng.annotations.*;

import java.time.Duration;
import java.util.List;

import static org.testng.Assert.*;

public class HomePageTest {

    private WebDriver webDriver;
    private HomePage homePage;

    @BeforeSuite
    public void beforeSuite() {
        WebDriverManager.edgedriver().setup();
    }
    @BeforeMethod
    public void setUp() {
        webDriver = new EdgeDriver();
        webDriver.get(LoginPage.URL);

        LoginPage loginPage = new LoginPage(webDriver);
        loginPage.setUsername("kts");
        loginPage.setPassword("kts");
        homePage = loginPage.submitLogin();
    }

    @AfterMethod
    public void tearDown() {
        webDriver.close();
    }

    @Test
    public void testLogOut() {
        homePage.logOut();
        assertEquals(webDriver.getCurrentUrl(), LoginPage.URL);
    }

    @Test
    public void featuredPosts() {
        int size = homePage.getFeaturedPostSize();
        assertEquals(size, 3);
    }

    @Test
    public void featuredPostsLinks() {
        List<String> links = homePage.getFeaturedPostLinks();
        assertNotNull(links, "There is no Featured Post Links");
        assertFalse(links.isEmpty(), "Featured Post Links is empty list.");
        links.forEach(link -> assertFalse(link.isEmpty(), "Link is empty."));
        links.forEach(link -> assertTrue(link.startsWith("https://"), "Link not starting with `https://`"));
    }

    @Test
    public void searchByTitle() {
        homePage.enterSearch("Business Intelligence Developer - Remote AZ");
        homePage.submitSearch();

        List<Post> posts = homePage.getJobPostResult();

        assertNotNull(posts);
        assertEquals(posts.size(), 1);
        assertEquals(posts.get(0).getPosition(), "Business Intelligence Developer - Remote AZ");
    }

    @Test
    public void searchByTitleLowercase() {
        homePage.enterSearch("Business Intelligence Developer - Remote AZ".toLowerCase());
        homePage.submitSearch();

        List<Post> posts = homePage.getJobPostResult();

        assertNotNull(posts);
        assertEquals(posts.size(), 1);
        assertEquals(posts.get(0).getPosition(), "Business Intelligence Developer - Remote AZ");
    }

    @Test
    public void numberOfPosts() {
        int fromAllPosts = homePage.getJobPostResult().size();
        int fromCount = homePage.getNumberOfPosts();

        assertEquals(fromAllPosts, fromCount);
    }

    @Test
    public void hybridAndRemote() {
        // given
        homePage.clickHybrid();
        homePage.clickRemote();

        // when
        homePage.search();

        // then
        assertTrue(homePage.isHybridSelected(), "Checkbox hybrid is not selected");
        assertTrue(homePage.isRemoteSelected(), "Checkbox remote is not selected");
    }

    @Test
    public void testSeniority() {
        homePage.selectSeniority("Senior");
        homePage.search();

        List<Post> posts = homePage.getJobPostResult();

        posts.forEach(post -> assertEquals(post.getSeniority(), "Senior"));
    }

    @Test
    public void cityAndRadius() {
        homePage.checkKragujevac();
        homePage.setRadius(45);

        homePage.search();

        List<Post> result = homePage.getJobPostResult();
        assertNotNull(result);
        assertEquals(result.size(), 1);
        assertEquals(result.get(0).getCity(), "Jagodina");
    }

    @Test
    @Parameters({"cityName"})
    public void filterByCity(String cityName) {
        homePage.checkCity(cityName);
        homePage.search();

        List<Post> result = homePage.getJobPostResult();
        assertNotNull(result);
        assertFalse(result.isEmpty());
        assertEquals(result.get(0).getCity(), cityName);
    }
}