package com.imi.polovniautomobili;

import io.github.bonigarcia.wdm.WebDriverManager;
import org.openqa.selenium.By;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.annotations.*;

import java.util.List;

import static org.testng.Assert.*;

public class HomePageTest {
    private static Logger LOGGER = LoggerFactory.getLogger(Logger.class);
    HomePage homePage;
    WebDriver driver;

    @BeforeSuite
    public void setUpDriverManager() {
        WebDriverManager.firefoxdriver().setup();
    }

    @BeforeMethod
    public void setUp() {
        driver = new FirefoxDriver();
        driver.get(LoginPage.URL);

        LoginPage loginPage = new LoginPage(driver);
        loginPage.setUsername("branko.arsic");
        loginPage.setPassword("pmf#imi");
        loginPage.submitCredentials();

        homePage = new HomePage(driver);

        driver.manage().window().maximize();
    }

    // 2. a
    @Test
    public void testNumberOfFeaturedCars() {
        // arrange

        // act
        List<FeaturedCar> featuredCars = homePage.getFeaturedCars();

        // assert
        assertNotNull(featuredCars);
        assertEquals(featuredCars.size(), 3);
    }

    // 2. b
    @Test
    public void testPresenceOfImageOfFeaturedCars() {
        // arrange

        // act
        List<FeaturedCar> featuredCars = homePage.getFeaturedCars();

        // assert
        assertNotNull(featuredCars);
        for (FeaturedCar featuredCar : featuredCars) {
            assertNotNull(featuredCar.getImageSource());

            // Greska:
            //  java.lang.AssertionError: did not expect [] but found []
            // nam govori da nismo ocekivali da "actual" bude prayan string, ali da se to ipak dogodilo
            assertNotEquals(featuredCar.getImageSource(), "");
        }
    }

    // 3. a
    @Test
    public void testSearchByModel_Fiat500AUT0MATlC() {
        // arrange
        homePage.setModelName("Fiat 500 AUT0MATlC");

        // act
        HomePage currentPage = homePage.performSearch();
        List<Car> carResults = homePage.getSearchResults();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertEquals(carResults.size(), 1);
        Car result = carResults.get(0);

        assertEquals(result.getManufacturer().trim(), "Fiat");
        assertEquals(result.getModel().trim(), "Fiat 500 AUT0MATlC");
        assertEquals(result.getYearOfManufacture(), "2009");
        assertEquals(result.getPrice(), "9000");
    }

    // 3. a. i
    @Test
    public void testSearchByModel_Fiat500AUT0MATlC_lowercase() {
        // arrange
        homePage.setModelName("Fiat 500 AUT0MATlC".toLowerCase());

        // act
        HomePage currentPage = homePage.performSearch();
        List<Car> carResults = homePage.getSearchResults();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertEquals(carResults.size(), 1);
        Car result = carResults.get(0);

        assertEquals(result.getManufacturer().trim(), "Fiat");
        assertEquals(result.getModel().trim(), "Fiat 500 AUT0MATlC");
        assertEquals(result.getYearOfManufacture(), "2009");
        assertEquals(result.getPrice(), "9000");
    }

    // 3. b
    @Test
    public void testPartialSearchByModel() {
        // arrange
        homePage.setModelName("a3");

        // act
        HomePage currentPage = homePage.performSearch();
        List<Car> carResults = homePage.getSearchResults();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertEquals(carResults.size(), 1);
        Car result = carResults.get(0);

        assertEquals(result.getModel().trim(), "A3 S LINE MATIK F1 CH");
    }

    // 4. a + b
    @Test
    @Parameters({"manufacturer", "numOfResults"})  // run-ujemo desnim preko xml fajla
    public void testFilterByBodyStyle(String manufacturer, int numOfResults) {
        // arrange
        homePage.selectManufacturer(manufacturer);

        // act
        HomePage currentPage = homePage.performSearch();
        List<Car> carResults = homePage.getSearchResults();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertEquals(carResults.size(), numOfResults);

        for (Car car : carResults) {
            assertEquals(car.getManufacturer().trim(), manufacturer);
        }
    }

    // 4. c
    @Test
    public void testBodyStyleFilterSelection() {
        // arrange
        String bodyStyle = "Limuzina";
        homePage.selectBodyStyle(bodyStyle);

        // act
        HomePage currentPage = homePage.performSearch();
        List<String> selectedOptions = homePage.getSelectedOptionsForBodyStyle();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        LOGGER.info("Selected options: " + selectedOptions);

        assertEquals(selectedOptions.size(), 1);
        assertEquals(selectedOptions.get(0).trim(), bodyStyle);
    }

    // 4. c - drugi nacin
    @Test
    public void testBodyStyleFilterSelection_2() {
        // arrange
        String bodyStyle = "Limuzina";
        homePage.selectBodyStyle(bodyStyle);

        // act
        HomePage currentPage = homePage.performSearch();
        boolean isSelected = homePage.isLimuzinaSelectedAsBodyStyle();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertTrue(isSelected);
    }

    // 4. c - treci nacin
    @Test
    public void testBodyStyleFilterSelection_3() {
        // arrange
        String bodyStyle = "Limuzina";
        homePage.selectBodyStyle(bodyStyle);

        // act
        HomePage currentPage = homePage.performSearch();
        boolean isSelected = homePage.isSelected(bodyStyle);

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        assertTrue(isSelected);
    }

    // 5.
    @Test
    public void testChosenFiltersPrint() {
        // arrange
        homePage.setModelName("QV");
        homePage.selectManufacturer("Alfa Romeo");
        homePage.selectBodyStyle("Limuzina");
        homePage.setUpToPrice("80000");
        homePage.setFromManufactureYear("2005");

        // act
        HomePage currentPage = homePage.performSearch();
        List<String> params = homePage.getSearchParams();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        LOGGER.info("Params: " + params);

        assertEquals(params.get(0).trim(), "QV");
        assertEquals(params.get(1).trim(), "Alfa Romeo");
        assertEquals(params.get(2).trim(), "Limuzina");
        assertEquals(params.get(3).trim(), "80000");
        assertEquals(params.get(4).trim(), "2005");
    }

    // 6.
    @Test
    public void testSort() {
        // arrange
        homePage.selectSortType(SortType.ASC);

        // act
        HomePage currentPage = homePage.performSearch();
        List<Car> cars = homePage.getSearchResults();

        // assert
        assertNotNull(currentPage);
        assertTrue(currentPage.getUrl().contains(HomePage.URL));

        LOGGER.info("Cars: " + cars);

        float previousPrice = -1F;
        for (Car car : cars) {
            float price = Float.parseFloat(car.getPrice());
            if (price > previousPrice) {
                previousPrice = price;
            } else {
                fail(String.format(
                        "Current price (%f) is not greater then prious price (%f)", price, previousPrice));
            }
        }
    }

    // 7.
    @Test
    public void testPresenceOfCarImageInsideNonEmptyLink() {
        // arrange

        // act
        List<WebElement> imgs = homePage.getCarImages();

        // assert
        for (WebElement img : imgs) {
            assertTrue(homePage.isInsideLink(img));
        }
    }

    @AfterMethod
    public void tearDown() {
        driver.quit();
    }
}
