from utils import *

class Problem:

    def __init__(self, initial, goal, graph):
        self.initial = initial
        self.goal = goal
        self.graph = graph

    def actions(self, a):
        return self.graph.get(a).keys()

    def goal_test(self, state):
        #return state == self.goal
        return any(x==state for x in self.goal)
    
    def path_cost(self, cost_so_far, A, B):
        return cost_so_far + self.graph.get(A)[B]

class Graph:

    def __init__(self, dictionary, directed=True):
        self.dict = dictionary
        self.directed = directed
        if directed:
            self.make_directed()
        else:
            self.make_undirected()

    def make_undirected(self):
        "Make a digraph into an undirected graph by adding symmetric edges."
        keys = list(self.dict.keys())
        for a in keys:
            b_items = self.dict[a].items()
            for (b, distance) in b_items:
                self.connect(b, a, distance)

    def make_directed(self):
        "Add all nodes without neighbours as keys with empty dictionaries as values"
        keys = list(self.dict.keys())
        for a in keys:
            a_keys = list(self.dict[a].keys())
            for b in a_keys:
                self.dict.setdefault(b, {})

    def connect(self, a, b, distance):
        "Add a link from A to B of given distance, in one direction only."
        self.dict.setdefault(a, {})
        self.dict[a][b] = distance

    def nodes(self):
        "Return a list of nodes in the graph."
        return self.dict.keys()
        
    def get(self, a):
        self.dict.setdefault(a, {})
        return self.dict[a]
    
class Node:
    
    def __init__(self, state, parent=None, path_cost=0):
        "Create a search tree Node, derived from a parent by an action."
        self.state = state
        self.parent = parent
        if parent == None:
            self.depth = 0
        else:
            self.depth = parent.depth + 1
        self.path_cost = path_cost
            
    def expand(self, problem):
        "List the nodes reachable in one step from this node."
        exp_nodes = []
        neighbour_states = problem.actions(self.state)
        for x in neighbour_states:
            curr_cost = self.path_cost
            new_cost = problem.path_cost(curr_cost, self.state, x)
            new_node = Node(x, self, new_cost)
            exp_nodes.append(new_node)
        return exp_nodes
    
    def solution(self):
        "Return a list of nodes forming the path from the root to this node."
        sol_states = []
        node = self
        while node != None:
            sol_states.insert(0, node.state)
            node = node.parent
        return sol_states

def graph_search(problem, open_nodes):
    """Search through the successors of a problem to find a goal.
    The argument open_nodes should be an empty queue."""
    explored = [problem.initial]
    open_nodes.append(Node(problem.initial))
    while len(open_nodes) > 0:
        node = open_nodes.pop()
        if problem.goal_test(node.state):
            #print "Path cost: %d" % node.path_cost
            return node.solution()
        for child in node.expand(problem):
            if child.state not in explored:
                open_nodes.append(child)
                explored.append(child.state)
    return None                
    
def breadth_first_search(problem):
    return graph_search(problem, FIFOQueue())

def depth_first_search(problem):
    return graph_search(problem, [])

def uniform_cost_search(problem):
    return graph_search(problem, SortedQueue())

def depth_limited_search(problem, limit):
    open_nodes = []
    explored = [problem.initial]
    open_nodes.append(Node(problem.initial))
    while len(open_nodes) > 0:
        node = open_nodes.pop()
        if problem.goal_test(node.state):
            return node.solution()
        if node.depth < limit:
            for child in node.expand(problem):
                if child.state not in explored:
                    open_nodes.append(child)
                    explored.append(child.state)
    return None  
    
def iterative_deepening_search(problem):
    for depth in range(len(problem.graph.nodes())):
        result = depth_limited_search(problem, depth)
        if result != None:
            return result
    return None
        
        
    