% nije function fajl
1;


function [errout,xo,to,Uo] = heatFTCS(nt,nx,alpha,L,tmax,errPlots)
% heatBTCS Solve 1D heat equation with the BTCS scheme
%
% Synopsis: heatFTCS
%             heatFTCS(nt)
%             heatFTCS(nt,nx)
%             heatFTCS(nt,nx,alpha)
%             heatFTCS(nt,nx,alpha,L)
%             heatFTCS(nt,nx,alpha,L,tmax)
%             heatFTCS(nt,nx,alpha,L,tmax,errPlots)
%             err = heatFTCS(...)
%             [err,x,t,U] = heatFTCS(...)
%
% Input:   nt = number of steps. Default: nt = 10;
%          nx = number of mesh points in x direction. Default: nx=20
%          alpha = diffusivity. Default: alpha = 0.1
%          L = length of the domain. Default: L = 1;
%          tmax = maximum time for the simulation. Default: tmax = 0.5
%          errPlots = flag (1 or 0) to control whether plots of the solution
%                     and the error at the last time step are created.
%                     Default: errPlots = 1 (make the plots)
%
% Output: err = L2 norm of error evaluated at the spatial nodes on last time step
%          x = location of finite difference nodes
%          t = values of time at which solution is obtained (time nodes)
%          U = matrix of solutions: U(:,j) is U(x) at t = t(j)
if nargin<1,  nt = 10;      end
if nargin<2,  nx = 20;      end
if nargin<3,  alpha = 0.1;  end
if nargin<4,  L = 1;        end
if nargin<5,  tmax = 0.5;   end
if nargin<6,  errPlots=1;   end

% --- Compute mesh spacing and time step
dx = L/(nx-1);
dt = tmax/(nt-1);
r = alpha*dt/dx^2;      r2 = 1 - 2*r;

r
r2

% --- Create arrays to save data for export
x = linspace(0,L,nx)';
t = linspace(0,tmax,nt);
U = zeros(nx,nt);

% --- Set IC and BC
U(:,1) = sin(pi*x/L);     %  implies u0 = 0; uL = 0;
u0 = 0;   uL = 0;         %  needed to apply BC inside time step loop

% --- Loop over time steps
for m=2:nt
	for i=2:nx-1
		U(i,m) = r*U(i-1,m-1) + r2*U(i,m-1) + r*U(i+1,m-1);
	end
end

U

% --- Compare with exact solution at end of simulation
ue = sin(pi*x/L)*exp(-t(nt)*alpha*(pi/L)^2);
err = norm(U(:,nt)-ue);

% --- Set values of optional output variables
if nargout>0, errout = err;                 end
if nargout>1, xo = x; to = t; Uo = U; end

% --- Plot error in solution at the last time step
if ~errPlots, return; end
fprintf('\nNorm of error = %12.3e at t = %8.3f\n',err,t(nt));
fprintf('\tdt, dx, r = %12.3e %12.3e      %8.3f\n',dt,dx,r);
figure; plot(x,U(:,nt),'o--',x,ue,'-');      xlabel('x'); ylabel('u');
legend('FTCS','Exact');
figure; plot(x,U(:,nt)-ue,'o--');            xlabel('x'); ylabel('u - u_e');
endfunction




function [errout,xo,to,Uo] = heatBTCS(nt,nx,alpha,L,tmax,errPlots)
% heatBTCS Solve 1D heat equation with the BTCS scheme
%
% Synopsis:	heatBTCS
% 	                heatBTCS(nt)
%      	        heatBTCS(nt,nx)
%   	                heatBTCS(nt,nx,alpha)
% 	                heatBTCS(nt,nx,alpha,L)
%          		heatBTCS(nt,nx,alpha,L,tmax)
%			heatBTCS(nt,nx,alpha,L,tmax,errPlots)
%			err = heatBTCS(...)
%			[err,x,t,U] = heatBTCS(...)
%
% Input: 		nt = number of steps. Default: nt = 10;
%           		nx = number of mesh points in x direction. Default: nx=20
%           		alpha = diffusion coefficient. Default: alpha = 0.1
%           		L = length of the domain. Default: L = 1;
%           		tmax = maximum time for the simulation. Default: tmax = 0.5
%           		errPlots = flag (1 or 0) to control whether plots of the solution
%                     	and the error at the last time step are created.
%                     	Default: errPlots = 1 (make the plots)
%
% Output: err = L2 norm of error evaluated at the spatial nodes on last time step
%          x = location of finite difference nodes
%          t = values of time at which solution is obtained (time nodes)
%          U = matrix of solutions: U(:,j) is U(x) at t = t(j)
if nargin<1,  nt = 10;      end
if nargin<2,  nx = 20;      end
if nargin<3,  alpha = 0.1;  end   %  diffusivity
if nargin<4,  L = 1;        end
if nargin<5,  tmax = 0.5;   end
if nargin<6,  errPlots=1;   end   %  flag: no error plots if errPlots=0

% --- Compute mesh spacing and time step
dx = L/(nx-1);
dt = tmax/(nt-1);

% --- Create arrays to save data for export
x = linspace(0,L,nx);
t = linspace(0,tmax,nt);
U = zeros(nx,nt);

% --- Set IC and BC
U(:,1) = sin(pi*x/L);     %  implies u0 = 0;  uL = 0;
u0 = 0;   uL = 0;

% --- Coefficients of the tridiagonal system
a = (-alpha/dx^2)*ones(nx,1);       % subdiagonal a: coefficients of phi(i-1)
c = a;                              % superdiagonal c: coefficients of phi(i+1)
b = (1/dt)*ones(nx,1) - 2*a;        % diagonal b: coefficients of phi(i)
b(1) = 1;      c(1) = 0;            % Fix coefficients of boundary nodes
b(end) = 1;    a(end) = 0;
[e,f] = tridiagLU(a,b,c);           % Get LU factorization of coefficient matrix

% --- Loop over time steps
for m=2:nt
	d = U(:,m-1)/dt;                               %  update right hand side
	d(1) = u0; d(end) = uL;                        %  overwrite BC values
  	U(:,m) = tridiagLUsolve(d,a,e,f,U(:,m-1));     %  solve the system
end

% --- Compare with exact solution at end of simulation
ue = sin(pi*x/L)*exp(-t(nt)*alpha*(pi/L)^2);
err = norm( U(:,nt) - transpose(ue) );

% --- Set values of optional output variables
if nargout>0, errout = err;           end
if nargout>1, xo = x; to = t; Uo = U; end

% --- Plot error in solution at the last time step
if ~errPlots, return; end
fprintf('\nNorm of error = %12.3e at t = %8.3f\n',err,t(nt));
figure; plot(x,U(:,nt),'o--',x,ue,'-');      xlabel('x'); ylabel('u');
legend('BTCS','Exact');
endfunction



function [errout,xo,to,Uo] = heatCN(nt,nx,alpha,L,tmax,errPlots)
% heatCN Solve 1D heat equation with the Crank-Nicolson scheme
%
% Synopsis: heatCN
%             heatCN(nt)
%             heatCN(nt,nx)
%             heatCN(nt,nx,alpha)
%             heatCN(nt,nx,alpha,L)
%             heatCN(nt,nx,alpha,L,tmax)
%             heatCN(nt,nx,alpha,L,tmax,errPlots)
%             err = heatCN(...)
%             [err,x,t,U] = heatCN(...)
%
% Input: nt = number of steps. Default: nt = 10;
%          nx = number of mesh points in x direction. Default: nx=20
%          alpha = diffusion coefficient. Default: alpha = 0.1
%          L = length of the domain. Default: L = 1;
%          tmax = maximum time for the simulation. Default: tmax = 0.5
%          errPlots = flag (1 or 0) to control whether plots of the solution
%                      and the error at the last time step are created.
%                      Default: errPlots = 1 (make the plots)
%
% Output: err = L2 norm of error evaluated at the spatial nodes on last time step
%          x = location of finite difference nodes
%          t = values of time at which solution is obtained (time nodes)
%          U = matrix of solutions: U(:,j) is U(x) at t = t(j)
if nargin<1,  nt = 10;      end
if nargin<2,  nx = 20;      end
if nargin<3,  alpha = 0.1;  end   %  diffusivity
if nargin<4,  L = 1;        end
if nargin<5,  tmax = 0.5;   end
if nargin<6,  errPlots=1;   end   %  flag: no error plots if errPlots=0
% --- Compute mesh spacing and time step
dx = L/(nx-1);
dt = tmax/(nt-1);
% --- Create arrays to save data for export
x = linspace(0,L,nx);
t = linspace(0,tmax,nt);
U = zeros(nx,nt);
% --- Set IC and BC
U(:,1) = sin(pi*x/L);     %  implies u0 = 0; uL = 0;
u0 = 0;   uL = 0;         %  needed to apply BC inside time step loop
% --- Coefficients of the tridiagonal system
a = (-alpha/2/dx^2)*ones(nx,1);       % subdiagonal a: coefficients of phi(i-1)
c = a;                                % superdiagonal c: coefficients of phi(i+1)
b = (1/dt)*ones(nx,1) - (a+c);        % diagonal b: coefficients of phi(i)
b(1) = 1;      c(1) = 0;              % Fix coefficients of boundary nodes
b(end) = 1;    a(end) = 0;
[e,f] = tridiagLU(a,b,c);             % Get LU factorization of coefficient matrix
% --- Loop over time steps
for m=2:nt
  % Right hand side includes time derivative and CN terms
  d = U(:,m-1)/dt - [0; a(2:end-1).*U(1:end-2,m-1); 0] ...
                   + [0; (a(2:end-1)+c(2:end-1)).*U(2:end-1,m-1); 0] ...
                   - [0; c(2:end-1).*U(3:end,m-1);    0];
  d(1) = u0; d(end) = uL;                      % overwrite BC values
  U(:,m) = tridiagLUsolve(d,a,e,f,U(:,m-1)); % solve the system
end
% --- Compare with exact solution at end of simulation
ue = sin(pi*x/L)*exp(-t(nt)*alpha*(pi/L)^2);
err = norm(U(:,nt)-ue);
% --- Set values of optional output variables
if nargout>0, errout = err;                 end
if nargout>1, xo = x; to = t; Uo = U; end
% --- Plot error in solution at the last time step
if ~errPlots, return; end
fprintf('\nNorm of error = %12.3e at t = %8.3f\n',err,t(nt));
fprintf('\tdt, dx, r = %12.3e %12.3e      %8.3f\n',dt,dx,r);
figure; plot(x,U(:,nt),'o--',x,ue,'-');      xlabel('x'); ylabel('u');
legend('FTCS','Exact');
figure; plot(x,U(:,nt)-ue,'o--');            xlabel('x'); ylabel('u - u_e');
endfunction



function showTheat(x,t,T,pflag)
% showTheat Plot and print solutions to the heat equation
%
% Synopsis: showTheat(x,t,T)
%           showTheat(x,t,T,pflag)
%
% Input: x = vector of positions at which temperature is known
%            t = vector of times at which solution is to be evaluated
%            T = matrix of T(x,t) values. T(:,t(j)) is T(x) at time t(j)
%            pflag = flag to control printing of results.
%                   Default: pflag = 0, do not print results
% --- Define string matrix of line styles that can be reused.
%      nsymb is the total number of line style strings (plot symbols)
%      In plot loop the statement isymb = 1 + rem(j-1,nsymb) is an
%      index in the range 1 <= isymb <= nsymb
lineSymb = strvcat('b-.','k-o','m-v','b-s','k*-','m-d','b-+','k-<','m-h');
nsymb = size(lineSymb,1);
% --- Plot T(x,t): each time is a different curve with different symbol
for j=1:length(t)
	hold on
	isymb = 1 + rem(j-1,nsymb);         % cyclic index for line styles.
	plot(x,T(:,j),lineSymb(isymb,:));
	% --- Build string matrix for legend. Each row is a legend string.
	s = sprintf('t = %-4.2f',t(j));
	if j==1
    	legstr = s;
  	else
    	legstr = strvcat(legstr,s);
  	end
end
hold off
legend(legstr,2);    xlabel('x');   ylabel('T(x,t)');
endfunction


function [e,f] = tridiagLU(a,b,c)
% tridiagLU Obtain the LU factorization of a tridiagonal matrix
%
% Synopsis: [e,f] = tridiag(a,b,c)
%
% Input: a,b,c = vectors defining the tridiagonal matrix. a is the
%                subdiagonal, b is the main diagonal, and c is the superdiagonal
%
% Output: e,f = vectors defining the L and U factors of the tridiagonal matrix
n = length(a);
e = zeros(n,1);  f = e;
e(1) = b(1);
f(1) = c(1)/b(1);
for i=2:n
	e(i) = b(i) - a(i)*f(i-1);
	f(i) = c(i)/e(i);
end
endfunction


function v = tridiagLUsolve(d,a,e,f,v)
% tridiagLUsolve Solve (LU)*v = d where L and U are LU factors of a tridiagonal
%                matrix
%
% Synopsis: v = tridiagLUsolve(d,e,f)
%           v = tridiagLUsolve(d,e,f,v)
%
% Input:    d = right hand side vector of the system of equatoins
%         e,f = vectors defining the L and U factors of the tridiagonal matrix.
%               e and f are obtained with the tridiagLU function
%           v = solution vector. If v is supplied, the elements of v are over-
%               written (thereby saving the memory allocation step). If v is not
%               supplied, it is created. v is used as a scratch vector in the
%               forward solve.
%
% Output: v = solution vector
n = length(d);
if nargin<5, v = zeros(n,1);    end
% --- Forward substitution to solve L*w = d
v(1) = d(1)/e(1);
for i=2:n
	v(i) = (d(i) - a(i)*v(i-1))/e(i);
end
% --- Backward substitution to solve U*v = w
for i=n-1:-1:1
	v(i) = v(i) - f(i)*v(i+1);
end
endfunction

