import pybullet as p
import pybullet_data
import cv2
import numpy as np
import time

# --------------------------------------------
# 1️⃣ Povezivanje sa PyBullet GUI okruženjem
# --------------------------------------------
p.connect(p.GUI)
p.setAdditionalSearchPath(pybullet_data.getDataPath())  # omogućava PyBullet-u da nađe URDF fajlove
p.resetSimulation()
p.setGravity(0, 0, -9.81)

# --------------------------------------------
# 2️⃣ Učitavanje ravne površine i KUKA robota
# --------------------------------------------
plane_id = p.loadURDF("plane.urdf")
robot_id = p.loadURDF("kuka_iiwa/model.urdf", [0, 0, 0], useFixedBase=True)

# --------------------------------------------
# 3️⃣ Učitavanje ciljnog objekta (virtuelni EV konektor)
# --------------------------------------------
connector_id = p.loadURDF("sphere_small.urdf", [0.5, 0.2, 0.05])


# --------------------------------------------
# 4️⃣ Funkcija za virtuelnu kameru
# --------------------------------------------
def get_camera_image():
    # Dobavljanje informacija iz debug kamere
    width, height, view_matrix, proj_matrix, _, _, _, _, _, _, _, _ = p.getDebugVisualizerCamera()
    img_arr = p.getCameraImage(320, 240)
    rgb = np.reshape(img_arr[2], (240, 320, 4))[:, :, :3]
    return cv2.cvtColor(rgb, cv2.COLOR_RGB2BGR)


# --------------------------------------------
# 5️⃣ Glavna petlja simulacije
# --------------------------------------------
for step in range(1000):
    # 5a️⃣ Uhvati sliku iz virtuelne kamere
    frame = get_camera_image()

    # 5b️⃣ Detekcija “konektora” po boji (plavi objekat)
    hsv = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV)
    mask = cv2.inRange(hsv, (100, 50, 50), (130, 255, 255))  # plava boja

    # 5c️⃣ Pronađi centar detektovanog objekta
    contours, _ = cv2.findContours(mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
    if contours:
        c = max(contours, key=cv2.contourArea)
        M = cv2.moments(c)
        if M["m00"] > 0:
            cx = int(M["m10"] / M["m00"])
            cy = int(M["m01"] / M["m00"])
            print(f"Detektovan konektor na slici: ({cx}, {cy})")

            # ❗ Ovde možeš dodati inverse kinematics da ruka ide ka konektoru
            # primer: p.calculateInverseKinematics(robot_id, endEffectorIndex, targetPos)

    # 5d️⃣ Prikaz slike sa maskom u OpenCV prozoru
    cv2.imshow("Camera View", frame)
    cv2.imshow("Mask", mask)

    # Korak simulacije
    p.stepSimulation()
    if cv2.waitKey(1) & 0xFF == ord('q'):
        break
    time.sleep(1. / 240.)

# --------------------------------------------
# 6️⃣ Kraj simulacije
# --------------------------------------------
cv2.destroyAllWindows()
p.disconnect()
